<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models\User;

use Erlage\Photogram\System;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Reflection\TraitReflectionClassName;

final class UserValidator
{
    use TraitReflectionClassName;

    public static function username(string $username): string
    {
        if (\strlen($username) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_USER_USERNAME))
        {
            return ResponseConstants::D_ERROR_USER_USERNAME_MAX_LEN_MSG;
        }

        if (\strlen($username) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_USER_USERNAME))
        {
            return ResponseConstants::D_ERROR_USER_USERNAME_MIN_LEN_MSG;
        }

        if ( ! \ctype_alpha($username[0]))
        {
            return ResponseConstants::D_ERROR_USER_USERNAME_STARTS_WITH_ALHPABET_MSG;
        }

        if (\preg_match('/[^a-z_\.0-9]/i', $username))
        {
            return ResponseConstants::D_ERROR_USER_USERNAME_ALLOWED_CHARACTERS_ONLY_MSG;
        }

        if (null != (new Query())
            -> from(UserTable::TABLE_NAME)
            -> where(UserTable::USERNAME, $username)
            -> selectOne()
        ) {
            return ResponseConstants::D_ERROR_USER_USERNAME_NOT_AVAILABLE_MSG;
        }

        return SystemConstants::OK;
    }

    public static function email(string $email): string
    {
        if (\strlen($email) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_USER_EMAIL))
        {
            return ResponseConstants::D_ERROR_USER_EMAIL_MAX_LEN_MSG;
        }

        if (\strlen($email) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_USER_EMAIL))
        {
            return ResponseConstants::D_ERROR_USER_EMAIL_MIN_LEN_MSG;
        }

        if (false === \filter_var($email, FILTER_VALIDATE_EMAIL))
        {
            return ResponseConstants::D_ERROR_USER_EMAIL_INVALID_FORMAT_MSG;
        }

        if (null !== (new Query())
            -> from(UserTable::TABLE_NAME)
            -> where(UserTable::EMAIL, $email)
            -> selectOne()
        ) {
            return ResponseConstants::D_ERROR_USER_EMAIL_NOT_AVAILABLE_MSG;
        }

        return SystemConstants::OK;
    }

    public static function password(string $password): string
    {
        if (\strlen($password) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_USER_PASSWORD))
        {
            return ResponseConstants::D_ERROR_USER_PASSWORD_MAX_LEN_MSG;
        }

        if (\strlen($password) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_USER_PASSWORD))
        {
            return ResponseConstants::D_ERROR_USER_PASSWORD_MIN_LEN_MSG;
        }

        return SystemConstants::OK;
    }

    public static function displayName(string $displayName): string
    {
        if (\strlen($displayName) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_USER_DISPLAY_NAME))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_NAME_MAX_LEN_MSG;
        }

        if (\strlen($displayName) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_USER_DISPLAY_NAME))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_NAME_MIN_LEN_MSG;
        }

        return SystemConstants::OK;
    }

    public static function displayBio(string $displayBio): string
    {
        if (\strlen($displayBio) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_USER_DISPLAY_BIO))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_BIO_MAX_LEN_MSG;
        }

        if (\strlen($displayBio) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_USER_DISPLAY_BIO))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_BIO_MIN_LEN_MSG;
        }

        return SystemConstants::OK;
    }

    public static function displayWeb(string $displayWeb): string
    {
        if (\strlen($displayWeb) > Settings::getInt(SystemConstants::SETTING_MAX_LEN_USER_DISPLAY_WEB))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_WEB_MAX_LEN_MSG;
        }

        if (\strlen($displayWeb) < Settings::getInt(SystemConstants::SETTING_MIN_LEN_USER_DISPLAY_WEB))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_WEB_MIN_LEN_MSG;
        }

        if (false === \filter_var($displayWeb, FILTER_VALIDATE_URL) && 0 !== \strlen($displayWeb))
        {
            return ResponseConstants::D_ERROR_USER_DISPLAY_WEB_INVALID_FORMAT_MSG;
        }

        if (0 !== \strlen($displayWeb))
        {
            if ( ! System::str_starts_with($displayWeb, 'http') && ! System::str_starts_with($displayWeb, 'https'))
            {
                return ResponseConstants::D_ERROR_USER_DISPLAY_WEB_INVALID_FORMAT_MSG;
            }
        }

        return SystemConstants::OK;
    }

    public static function metaIsPrivate(string $metaIsPrivate): string
    {
        if ( ! \in_array($metaIsPrivate, array(UserEnum::META_IS_PRIVATE_YES, UserEnum::META_IS_PRIVATE_NO)))
        {
            return ResponseConstants::ERROR_BAD_REQUEST_MSG;
        }

        return SystemConstants::OK;
    }
}
